package gov.cms.grouper.snf.component.v100.logic.nursing;

import static gov.cms.grouper.snf.component.v100.TestUtil.of;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.spy;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import gov.cms.grouper.snf.component.v100.TestUtil;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.enums.NursingCmg;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.util.ClaimInfo;

public class SpecialCareTest {

  public static SpecialCare make(ClaimInfo claim) {
    return make(claim, null);
  }

  public static SpecialCare make(ClaimInfo claim, Boolean depressed) {
    if (depressed != null) {
      int val = 12;
      if (depressed == false) {
        val = 0;
      }
      doReturn(val).when(claim).getAssessmentValue(Rai300.D0600);
    }
    final SpecialCare logic = new SpecialCare(claim);
    SpecialCare spy = spy(logic);


    return spy;
  }


  @Test
  public void evaluateSpecialCareHighTest() {
    ClaimInfo claim = of(4);
    SpecialCare sc = make(claim, true);
    doReturn(true).when(sc).isSpecialCareHighApplicable();
    NursingCmg expectedCmg = NursingCmg.HDE2;
    NursingCmg actualCmg = sc.evaluateSpecialCareHigh();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(4);
    sc = make(claim, false);
    doReturn(true).when(sc).isSpecialCareHighApplicable();
    expectedCmg = NursingCmg.HDE1;
    actualCmg = sc.evaluateSpecialCareHigh();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(7);
    sc = make(claim, false);
    doReturn(true).when(sc).isSpecialCareHighApplicable();
    expectedCmg = NursingCmg.HBC1;
    actualCmg = sc.evaluateSpecialCareHigh();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(7);
    sc = make(claim, true);
    doReturn(true).when(sc).isSpecialCareHighApplicable();
    expectedCmg = NursingCmg.HBC2;
    actualCmg = sc.evaluateSpecialCareHigh();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(15);
    sc = make(claim, true);
    doReturn(false).when(sc).isSpecialCareHighApplicable();
    Assertions.assertNull(sc.evaluateSpecialCareHigh());

    // Called evaluateClinicalComplex
    claim = of(15);
    sc = make(claim, true);
    doReturn(true).when(sc).isSpecialCareHighApplicable();
    Assertions.assertEquals(NursingCmg.CA2, sc.evaluateSpecialCareHigh());
  }

  @Test
  public void isSpecialCareHighTest() {
    // isQuadriplegia
    ClaimInfo claim = of(1);
    doReturn(true).when(claim).isCheckedAndNotNull(Rai300.I5100);
    SpecialCare sc = make(claim);
    Assertions.assertTrue(sc.isSpecialCareHighApplicable());


    // isParenteral
    claim = of(4);
    doReturn(true).when(claim)
        .isAnyAssessmentValuesPresent(SnfUtils.toSet(Rai300.K0510A1, Rai300.K0510A2));
    sc = make(claim);
    Assertions.assertTrue(sc.isSpecialCareHighApplicable());

    claim = of(3);
    sc = make(claim);
    Assertions.assertFalse(sc.isSpecialCareHighApplicable());
  }

  @Test
  public void evaluateSpecialCareLowTest() {
    ClaimInfo claim = of(3);
    SpecialCare sc = make(claim, true);
    doReturn(true).when(sc).isSpecialCareLowApplicable();
    NursingCmg expectedCmg = NursingCmg.LDE2;
    NursingCmg actualCmg = sc.evaluateSpecialCareLow();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(3);
    sc = make(claim, false);
    doReturn(true).when(sc).isSpecialCareLowApplicable();
    expectedCmg = NursingCmg.LDE1;
    actualCmg = sc.evaluateSpecialCareLow();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(7);
    sc = make(claim, false);
    doReturn(true).when(sc).isSpecialCareLowApplicable();
    expectedCmg = NursingCmg.LBC1;
    actualCmg = sc.evaluateSpecialCareLow();
    Assertions.assertEquals(expectedCmg, actualCmg);

    claim = of(7);
    sc = make(claim, true);
    doReturn(true).when(sc).isSpecialCareLowApplicable();
    expectedCmg = NursingCmg.LBC2;
    actualCmg = sc.evaluateSpecialCareLow();
    Assertions.assertEquals(expectedCmg, actualCmg);

    NursingCmg expected = NursingCmg.CDE2;

    claim = of(15);
    sc = make(claim, false);
    doReturn(true).when(sc).isSpecialCareLowApplicable();
    doReturn(expected).when(sc).evaluateClinicallyComplex(anyBoolean());
    NursingCmg actual = sc.evaluateSpecialCareLow();
    Assertions.assertEquals(expected, actual);

    claim = of(15);
    sc = make(claim, false);
    doReturn(false).when(sc).isSpecialCareLowApplicable();
    doReturn(expected).when(sc).evaluateClinicallyComplex(anyBoolean());
    actual = sc.evaluateSpecialCareLow();
    Assertions.assertEquals(expected, actual);
  }


  public static SpecialCare make(boolean cerebralPalsy, boolean multiSclerosis, boolean parkinson,
      boolean respiratoryOxyTherapy, boolean feedingTube, boolean stage2Ulcer,
      boolean stage3Or4Ulcer, boolean twoOrMoreVenousArterialUlcers,
      boolean stage2UlcerAndVenousArterialUlcer, boolean foot, boolean radiation, boolean dialysis,
      int functionScore) {
    ClaimInfo claim = of(functionScore);
    SpecialCare sc = make(claim, true);

    doReturn(cerebralPalsy).when(sc).isCerebralPalsy();
    doReturn(multiSclerosis).when(sc).isMultiSclerosis();
    doReturn(parkinson).when(sc).isParkinson();
    doReturn(feedingTube).when(sc).isFeedingTube();
    doReturn(stage2UlcerAndVenousArterialUlcer).when(sc).isStage2UlcerAndVenousArterialUlcer();
    doReturn(respiratoryOxyTherapy).when(sc).isRespiratoryOxyTherapy();
    doReturn(stage2Ulcer).when(sc).isStage2Ulcer();
    doReturn(stage3Or4Ulcer).when(sc).isStage3Or4Ulcer();
    doReturn(twoOrMoreVenousArterialUlcers).when(sc).isTwoOrMoreVenousArterialUlcers();
    doReturn(foot).when(sc).isFoot();

    // dialysis
    doReturn(dialysis).when(claim).isCheckedAndNotNull(Rai300.O0100J2);
    doReturn(radiation).when(claim).isCheckedAndNotNull(Rai300.O0100B2);

    return sc;
  }



  @Test
  public void isSpecialCareLowTest() {

    SpecialCare sc =
        make(true, true, true, false, false, false, false, false, false, false, false, false, 1);
    Assertions.assertTrue(sc.isSpecialCareLowApplicable());


    sc = make(false, false, false, false, false, false, false, false, false, false, false, false,
        3);
    Assertions.assertFalse(sc.isSpecialCareLowApplicable());

    sc = make(false, false, false, false, false, false, true, false, false, false, false, false, 3);
    Assertions.assertTrue(sc.isSpecialCareLowApplicable());

    int fs = Integer.MIN_VALUE;
    int size = 12;
    Boolean[] values = new Boolean[size];
    Arrays.fill(values, false);
    List<Boolean> params = Arrays.asList(values);
    for (int i = 0; i < 12; i++) {
      params = Arrays.asList(values);
      params.set(i, true);
      sc = make(params.get(0), params.get(1), params.get(2), params.get(3), params.get(4),
          params.get(5), params.get(6), params.get(7), params.get(8), params.get(9), params.get(10),
          params.get(11), fs);
      Assertions.assertTrue(sc.isSpecialCareLowApplicable());
    }

  }


  @Test
  public void testIsDepressed() {
    int version = 100;
    List<Assessment> assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 0),
        new Assessment("", Rai300.D0600.name(), 0));
    ClaimInfo claim = ClaimInfo.of(version, false, assessments);
    SpecialCare sc = new SpecialCare(claim);

    boolean expected = false;
    boolean actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 9),
        new Assessment("", Rai300.D0600.name(), 9));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = false;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 10),
        new Assessment("", Rai300.D0600.name(), 9));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = true;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 99),
        new Assessment("", Rai300.D0600.name(), 9));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = false;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 100),
        new Assessment("", Rai300.D0600.name(), 9));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = true;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 100),
        new Assessment("", Rai300.D0600.name(), 19));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = true;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 99),
        new Assessment("", Rai300.D0600.name(), 10));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = true;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

    assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 0),
        new Assessment("", Rai300.D0600.name(), 99));
    claim = ClaimInfo.of(version, false, assessments);
    sc = new SpecialCare(claim);
    expected = true;
    actual = sc.isDepressed();
    Assertions.assertEquals(expected, actual);

  }

  public static SpecialCare make(boolean isClinicallyComplexApplicable, int functionScore,
      boolean isDepressed) {
    ClaimInfo claim = of(functionScore);
    SpecialCare sc = make(claim, isDepressed);
    doReturn(isClinicallyComplexApplicable).when(sc).isClinicallyComplexApplicable();


    return sc;
  }

  @Test
  public void testEvaluateClinicallyComplex() {
    SpecialCare sc = make(true, 0, true);
    Assertions.assertEquals(NursingCmg.CDE2, sc.evaluateClinicallyComplex(false));

    sc = make(true, 4, true);
    Assertions.assertEquals(NursingCmg.CDE2, sc.evaluateClinicallyComplex(false));

    sc = make(true, 5, false);
    Assertions.assertEquals(NursingCmg.CDE1, sc.evaluateClinicallyComplex(false));

    sc = make(true, 6, true);
    Assertions.assertEquals(NursingCmg.CBC2, sc.evaluateClinicallyComplex(false));

    sc = make(true, 8, true);
    Assertions.assertEquals(NursingCmg.CBC2, sc.evaluateClinicallyComplex(false));

    sc = make(true, 14, false);
    Assertions.assertEquals(NursingCmg.CBC1, sc.evaluateClinicallyComplex(false));

    sc = make(true, 15, true);
    Assertions.assertEquals(NursingCmg.CA2, sc.evaluateClinicallyComplex(false));

    sc = make(true, 16, false);
    Assertions.assertEquals(NursingCmg.CA1, sc.evaluateClinicallyComplex(false));

    sc = make(false, 6, true);
    Assertions.assertNull(sc.evaluateClinicallyComplex(false));
    Assertions.assertEquals(NursingCmg.CBC2, sc.evaluateClinicallyComplex(true));

    sc = make(true, 17, false);
    Assertions.assertNull(sc.evaluateClinicallyComplex(false));
  }



  public static SpecialCare make(List<Assessment> assessment, int functionScore) {
    ClaimInfo claim = of(assessment, functionScore);
    final SpecialCare logic = make(claim);
    SpecialCare spy = spy(logic);

    return spy;
  }

  @Test
  public void testIsClinicallyComplexApplicable() {
    // Pneumonia
    SpecialCare sc = make(Collections.singletonList(new Assessment("", Rai300.I2000.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    sc = make(Collections.singletonList(new Assessment("", Rai300.I2000.name(), 0)), 0);
    Assertions.assertFalse(sc.isClinicallyComplexApplicable());;

    // Hemiplagia/hemiparesis
    sc = make(Collections.singletonList(new Assessment("", Rai300.I4900.name(), 1)), 11);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    sc = make(Collections.singletonList(new Assessment("", Rai300.I4900.name(), 1)), 12);
    Assertions.assertFalse(sc.isClinicallyComplexApplicable());;
    // Open lesions
    sc = make(Arrays.asList(new Assessment("", Rai300.M1040D.name(), 1),
        new Assessment("", Rai300.M1200F.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    sc = make(Arrays.asList(new Assessment("", Rai300.M1040D.name(), 1),
        new Assessment("", Rai300.M1200F.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    sc = make(Arrays.asList(new Assessment("", Rai300.M1040D.name(), 1),
        new Assessment("", Rai300.M1200G.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    sc = make(Arrays.asList(new Assessment("", Rai300.M1040D.name(), 1),
        new Assessment("", Rai300.M1200H.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;


    sc = make(Arrays.asList(new Assessment("", Rai300.M1040E.name(), 1),
        new Assessment("", Rai300.M1200F.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    sc = make(Arrays.asList(new Assessment("", Rai300.M1040E.name(), 1),
        new Assessment("", Rai300.M1200H.name(), 1)), 0);
    sc = make(Arrays.asList(new Assessment("", Rai300.M1040E.name(), 1),
        new Assessment("", Rai300.M1200G.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;
    // Burns
    sc = make(Collections.singletonList(new Assessment("", Rai300.M1040F.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;


    // Chemotherapy while a resident
    sc = make(Collections.singletonList(new Assessment("", Rai300.O0100A2.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;

    // Oxygen therapy while a resident
    sc = make(Collections.singletonList(new Assessment("", Rai300.O0100C2.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;
    // IV Medications while a resident
    sc = make(Collections.singletonList(new Assessment("", Rai300.O0100H2.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;
    // Transfusions while a resident
    sc = make(Collections.singletonList(new Assessment("", Rai300.O0100I2.name(), 1)), 0);
    Assertions.assertTrue(sc.isClinicallyComplexApplicable());;
  }



  @Test
  public void testExec() {
    List<Assessment> assessments = Arrays.asList(new Assessment("", Rai300.D0300.name(), 1),
        new Assessment("", Rai300.N0350A.name(), 1), new Assessment("", Rai300.D0600.name(), 1),
        new Assessment("", Rai300.I2100.name(), 1));
    ClaimInfo claim = of(assessments, 7);

    SpecialCare care = make(claim);
    NursingCmg actual = care.exec();
    NursingCmg expected = NursingCmg.HBC1;
    Assertions.assertEquals(expected, actual);

    claim = of(TestUtil.getAll(1), 1);
    care = make(claim);
    actual = care.exec();
    expected = NursingCmg.HDE1;
    Assertions.assertEquals(expected, actual);

  }

}
